<?php

/**
 * Copyright(C) 2015 Markus Bröker<broeker.markus@googlemail.com>
 *
 */

namespace bfw\core;

use Smarty;

/**
 * Class View
 */
class View {

    /**
     * Statische Referenz auf die reale Template Engine, hier Smarty
     *
     * @var Smarty
     */
    private static $tpl = null;

    /**
     * @var Model
     */
    private $model;

    /**
     * @var Controller
     */
    private $controller;

    /**
     * View constructor.
     *
     * @param Controller $controller
     * @param Model $model
     */
    public function __construct(Controller $controller, Model $model) {
        $this->controller = $controller;
        $this->model = $model;

        self::$tpl = self::getTemplateEngine();

        $request = $controller->getRequest();
        $this->assign('controller', $request->get('controller', 'home'));
        $this->assign('action', $request->get('action', 'index'));
    }

    /**
     * <b>Liefert ein statisches Handle auf die aktuelle Template Engine zurück</b>
     *
     * Das Singleton Pattern garantiert genau eine Instanz der Template Engine im Speicher.
     *
     *  <b>Verschiedene Skripte können so nacheinander Variablen zuweisen und diese bei Bedarf auslesen</b>
     *
     * @return Smarty
     */
    private static function getTemplateEngine() {
        if (self::$tpl == null) {
            self::$tpl = new Smarty();

            $tpl = self::$tpl;

            $tpl->setCompileDir('./data/templates_c/');
            $tpl->setTemplateDir('./classes/bfw/mvc/view/');
        }

        return self::$tpl;
    }

    /**
     * @param $template
     * @return string
     */
    public function fetch($template) {
        return self::$tpl->fetch($template);
    }

    /**
     * <b>Anzeige der komplett gerenderten Seite mit dem Layout $layout</b>
     *
     * @param string $layout
     * @return int
     */
    public function display($layout = 'layout.tpl') {
        $request = $this->controller->getRequest();

        $controller = $this->controller;

        $response = $controller->getResponse();
        $this->assign('response', $response);

        $this->assign('request', $request);

        $model = $controller->getModel();
        $user = $model->getUser();
        $group = $model->getGroup();

        if ($request->getKey('isLoggedIn')) {
            $user->find($request->getKey('user_id'));
        }

        $this->assign('user', $user);
        $this->assign('group', $group);

        try {
            self::$tpl->display($layout);
        } catch (SmartyException $se) {
            $response->setStatusCode(404);

            self::$tpl->display('layout.tpl');
        }

        // Msg wieder auf Null setzen...
        $request->setKey('msg', '');

        // Fehler wieder auf Null setzen...
        $request->setKey('error', '');

        return http_response_code();
    }

    public function assign($key, $value) {
        self::$tpl->assign($key, $value);
    }

    /**
     * @param string $key
     * @return string
     */
    public function get($key) {
        return self::$tpl->getTemplateVars($key);
    }

}
