<?php

/**
 * Copyright(C) 2015 Markus Bröker<broeker.markus@googlemail.com>
 *
 */

namespace bfw\core;

use bfw\database\Database;
use Logger;

/**
 * Class Entity
 *
 * Entity besitzt einen State in $data
 *
 *
 * @method string getId()
 * @method setId($param)
 */
class Entity {

    protected $logger;
    protected $db;
    protected $table;
    protected $data;

    /**
     * @param $table
     */
    public function __construct($table) {
        $this->logger = Logger::getLogger(get_class($this));

        $this->db = Database::getInstance();

        $this->table = $table;
        $this->data = array();
    }

    /**
     * Magische Methode: Was soll beim Eintüten eingetütet werden?
     *
     * @return array
     */
    public function __sleep() {
        return array(
            'table',
            'data',
        );
    }

    /**
     * Magische Methode: Was soll nach dem Auspacken noch passieren?
     */
    public function __wakeup() {
        $this->logger = Logger::getLogger(get_class($this));
        $this->db = Database::getInstance();
    }

    /**
     * <b>Herzstück des Persistence-Frameworks</b>
     *
     * Dynamische Erzeugung von Getttern- und Settern
     *
     * Anstatt diese explizit zu pflegen und zu warten,
     * werden Diese dynamisch, on, the fly, erzeugt.
     *
     * <b>Der Vorteil: Nicht Konfigurieren, nicht Erzeugen. Einfach Nutzen...</b>
     *
     * @param $methodName
     * @param null $params
     * @return mixed
     */
    public function __call($methodName, $params = null) {
        $prefix = substr($methodName, 0, 3);

        $key = lcfirst(substr($methodName, 3));
        $key = strtolower(preg_replace('/([A-Z])/', "_$1", $key));

        if ($prefix == 'set') {
            $value = $params[0];

            // no data, no state, null
            if (!is_array($this->data)) {
                return null;
            }

            $this->data[$key] = $value;

            // fluent please!
            return $this;
        } else if ($prefix == 'get') {

            // no data, no state, null
            if (!is_array($this->data)) {
                return null;
            }

            if (array_key_exists($key, $this->data)) {
                return $this->data[$key];
            }
        }

        exit(sprintf('Unbekannte Methode %s::%s(%s, %s) wurde aufgerufen.', get_class($this), $methodName, $key, $prefix));
    }

    /**
     * <b>Vereinfachte Find Methode bezogen auf die jeweilige Instanz</b>
     *
     * @param $id
     * @return Model
     */
    public function find($id) {
        $this->logger->info(sprintf('%s(%d) ', __METHOD__, $id));

        if (($this->data = $this->db->find($this->table, $id)) == null) {
            return null;
        }

        return $this;
    }

    /**
     * <b>Die Findall Methode erzeugt typsicher den Supertyp</b>
     *
     * @param int $initial_id
     * @return Entity[]
     */
    public function findAll($initial_id = 1) {
        $this->logger->info(sprintf('%s(%d) ', __METHOD__, $initial_id));

        $rows = $this->db->findAll($this->table, $initial_id);

        if (count($rows) == 0) {
            return array();
        }

        $list = array();
        foreach ($rows as $row) {
            $item = new static();
            $item->data = $row;

            $list[] = $item;
        }

        $this->logger->info(sprintf('%s(): %d', __METHOD__, count($list)));

        return $list;
    }

    /**
     * <b>Vereinfachte FindByField Methode</b>
     *
     * @param $field
     * @param $value
     *
     * @return Entity
     */
    public function findByField($field, $value) {
        $this->logger->info(sprintf('%s(%s, %s) ', __METHOD__, $field, $value));

        $this->data = $this->db->findByField($this->table, $field, $value);

        // Keine Daten, keine Instanz
        if ($this->data == null) {
            return null;
        }

        return $this;
    }

    /**
     * <b>Vereinfachte FindAllByField Methode</b>
     *
     * Erzeugt on the fly ein array mit dem Supertyp der Klasseninstanz
     * @param $field
     * @param $value
     *
     * @return Entity[]
     */
    public function findAllByField($field, $value) {
        $this->logger->info(sprintf('%s(%s, %s) ', __METHOD__, $field, $value));

        $rows = $this->db->findAllByField($this->table, $field, $value);

        if ($rows == null) {
            return null;
        }

        $list = array();
        foreach ($rows as $row) {
            $item = new static();
            $item->data = $row;

            $list[] = $item;
        }

        return $list;
    }

    /**
     * <b>Vereinfachte persist Methode</b>
     *
     * Der Logger loggt nur die Nutzdaten, deswegen sieht das im Log identisch aus.
     *
     * Es handelt sich allerdings einmal um ein Objekt und einmal nur um ein Array.
     *
     * @return bool
     */
    public function persist() {
        $this->logger->info(sprintf('%s(%s)', __METHOD__, print_r($this->getData(), true)));

        return $this->db->persist($this->table, $this->getData());
    }

    public function getData() {
        return $this->data;
    }

    /**
     * <b>Vereinfachte store Methode</b>
     *
     * @return bool
     */
    public function store() {
        $this->logger->info(sprintf('%s(%d, %s)', __METHOD__, $this->getId(), print_r($this->getData(), true)));

        if ($this->getId() > 1) {
            return $this->db->store($this->table, $this->getId(), $this->getData());
        }

        return false;
    }

    /**
     * <b>Vereinfachte delete Methode</b>
     *
     * @return bool
     */
    public function delete() {
        $this->logger->info(sprintf('%s(%d) ', __METHOD__, $this->getId()));

        if ($this->getId() > 1) {
            return $this->db->delete($this->table, $this->getId());
        }

        return false;
    }

    /**
     * Experimental Feature
     *
     * @param $array
     * @return $this
     */
    public function merge($array) {
        $this->data = array_merge($this->data, $array);

        return $this;
    }

    /**
     * @return string
     */
    public function getLastError() {
        return $this->db->getLastError();
    }

    /**
     *
     * @param string $string
     * @return string
     */
    public function quote($string) {
        return $this->db->quote($string);
    }

}
