<?php

namespace bfw\mvc\controller {

    /**
     * Copyright(C) 2015 Markus Bröker<broeker.markus@googlemail.com>
     *
     */

    use bfw\Dispatcher;
    use bfw\mvc\model\Model;
    use bfw\mvc\model\TGroup;
    use bfw\mvc\model\TStatus;
    use bfw\mvc\model\TTicket;
    use bfw\mvc\model\TUser;

    class TicketsController extends Controller {
        private $user_id;
        private $group_id;
        private $currentTicket;

        /**
         * TicketsController constructor.
         *
         * @param Model $model
         * @param Request $request
         */
        public function __construct($model, $request) {
            parent::__construct($model, $request);

            $this->currentTicket = null;

            $this->user_id = $request->getKey('user_id', TUser::SYSTEM);
            $this->group_id = $request->getKey('group_id', TGroup::GUEST);
        }

        /**
         * <b>Diese Funktion kümmert sich um die Anfragen dieses Ticket-Controllers</b>
         *
         * @param $currentTicket
         * @param $user_id
         * @param $group_id
         */
        public function handleRequest($currentTicket, $user_id, $group_id) {
            $request = $this->getRequest();
            $engine = $this->getView();

            /**
             * @var Model
             */
            $tTicket = $this->getModel();

            $tUser = new TUser();
            $user = $tUser->find($user_id);

            /**
             * 'Cron-Job' ohne 'Cron': Alle Tickets werden bei jedem Zugriff geprüft
             *
             * Gibt es Tickets, die derzeit gesperrt sind? Befreie Sie! Free Tickets!!!
             * Falls Tickets frei sind, weise diese korrekt und fair wieder zu.
             */
            $tTicket->unlockTickets($tTicket);
            $tTicket->reassignFreeTickets();

            switch ($group_id) {
                case TGroup::SUPPORTER:
                    $tickets = $tTicket->findAllTicketsByUserid($user_id);
                    if ($tickets == null) {
                        $request->setKey('fehler', 'Es existieren derzeit keine Tickets für Sie!');
                    }

                    $users = array($tUser->findByUsername($user->getUsername()));
                    break;
                case TGroup::ADMIN:
                case TGroup::MANAGER:
                    $tickets = $tTicket->findAll();
                    $users = $tUser->findAll(false);
                    break;
                default:
                    $tickets = $tTicket->findAll();
                    $users = $tUser->findAll();
            }

            $tStatus = new TStatus();
            $statusItems = $tStatus->findAll();

            // erstelle eine neue Ticket-Instanz, falls es keine instanzierte gibt.
            if ($currentTicket == null) {
                $currentTicket = new TTicket();
            }

            $engine->assign('users', $users);
            $engine->assign('tStatus', $tStatus);
            $engine->assign('tickets', $tickets);
            $engine->assign('statusItems', $statusItems);
            $engine->assign('currentTicket', $currentTicket);
            $engine->assign('group_id', $group_id);
        }

        /**
         * @param $user_id
         * @return null|TTicket
         */
        public function processTickedId($user_id) {
            $request = $this->getRequest();
            $engine = $this->getView();

            $tUser = new TUser();
            $user = $tUser->find($user_id);

            if ($user == null) {
                $request->setKey('fehler', sprintf('Benutzer(%d) konnte nicht mehr gefunden werden.', $user_id));
                Dispatcher::route('/');
            }

            $tid = $request->get('tid');

            $tTicket = $this->getModel();
            $currentTicket = $tTicket->find($tid);

            if ($currentTicket == null) {
                $request->setKey('fehler', sprintf('Ticket Nr. %d konnte nicht gefunden werden.', $tid));

                return null;
            }

            if (!$currentTicket->hasPrivilege($currentTicket, $user, false)) {
                $request->setKey('fehler', sprintf('Ihnen fehlen die Berechtigungen für das Ticket Nr. %d.', $tid));

                return null;
            }

            return $currentTicket;
        }

        /**
         * <b>Ticket nach TID laden</b>
         *
         * @param $user_id
         *
         * @return null|TTicket
         */
        public function processLaden($user_id) {
            $request = $this->getRequest();
            $engine = $this->getView();

            $tid = $request->session('tid');

            $tTicket = $this->getModel();
            $currentTicket = $tTicket->find($tid);

            if ($currentTicket == null) {
                $request->setKey('fehler', sprintf('Ticket Nr. %d konnte nicht gefunden werden.', $tid));

                return null;
            }

            $tUser = new TUser();
            $user = $tUser->find($user_id);

            if ($user == null) {
                $request->setKey('fehler', sprintf('Benutzer(%d) konnte nicht mehr gefunden werden.', $user_id));

                return null;
            }

            $tTicket->lockTicket($currentTicket, $user);

            if (!$currentTicket->hasPrivilege($currentTicket, $user, true)) {
                $request->setKey('fehler', sprintf('Ihnen fehlen die Berechtigungen für das Ticket Nr. %d.', $tid));

                return null;
            }

            return $currentTicket;
        }

        /**
         * <b>Löschen von Tickets</b>
         *
         * @return bool
         */
        public function processLoeschen($user_id) {
            $request = $this->getRequest();
            $engine = $this->getView();

            $tid = $request->session('tid');

            $tTicket = $this->getModel();
            $currentTicket = $tTicket->find($tid);

            if ($currentTicket == null) {
                $request->setKey('fehler', sprintf('Nicht existierende Tickets(%d) können nicht gelöscht werden.', $tid));

                return false;
            }

            $tUser = new TUser();
            $user = $tUser->find($user_id);

            if ($user == null) {
                $request->setKey('fehler', sprintf('Benutzer(%d) konnte nicht mehr gefunden werden.', $user_id));

                return false;
            }

            if (!$currentTicket->hasPrivilege($currentTicket, $user, true)) {
                $request->setKey('fehler', sprintf('Ihnen fehlen die Berechtigungen für das Ticket Nr. %d.', $tid));

                return false;
            }

            if ($currentTicket->delete($tid)) {
                $request->setKey('msg', sprintf('Das Ticket mit der Nummer %d wurde gelöscht.', $tid));

                return true;
            }

            $request->setKey('fehler', sprintf('Das Ticket mit der Nummer %d konnte nicht gelöscht werden.', $tid));

            return false;
        }

        /**
         * @param $user_id
         * @param $group_id
         * @return Model|null
         */
        public function processEintragen($user_id, $group_id) {
            $request = $this->getRequest();
            $engine = $this->getView();

            $logger = Logger::getLogger('main');

            $tid = $request->session('tid');
            $status_id = $request->session('status', TStatus::OFFEN);

            $tTicket = $this->getModel();

            if ($tid != '') {
                $tTicket->find($tid, $user_id);
            }

            $tUser = new TUser();
            $user = $tUser->find($user_id);
            if ($user == null) {
                return null;
            }

            $theHolyUserId = $tTicket->getUser_Id();
            if ($group_id == TGroup::ADMIN || $group_id == TGroup::MANAGER) {
                $theHolyUserId = $request->session('user_id');
            }

            $subject = $request->session('subject');
            $message = $request->session('message');

            if (($subject == '') && ($message == '')) {
                $request->setKey('fehler', sprintf('Füllen Sie die Pflichtfelder aus.'));

                return null;
            }

            $tTicket->setUser_Id($theHolyUserId);
            $tTicket->setStatus_Id($status_id);
            $tTicket->setSubject($subject);
            $tTicket->setMessage($message);
            $tTicket->setLast_Access(date("Y-m-d H:i:s"));

            if ($tid == '') {
                if ($tTicket->persist($tTicket)) {
                    $request->setKey('msg', sprintf('Ein neues Ticket wurde erfolgreich eingetragen.'));
                    $logger->info(sprintf('Ticket wurde eingetragen: (%s)', print_r($tTicket->getData(), true)));

                    return null;
                }
            } else {
                if (!$tTicket->hasPrivilege($tTicket, $user, true)) {
                    $request->setKey('fehler', sprintf('Ihnen fehlen die Berechtigungen für das Ticket Nr. %d.', $tid));

                    return null;
                }

                if ($tTicket->getId() != 1) {
                    if ($tTicket->store($tid, $tTicket)) {
                        $request->setKey('msg', 'Das Ticket wurde erfolgreich aktualisiert.');
                        $logger->info(sprintf('Ticket wurde aktualisiert: (%s)', print_r($tTicket->getData(), true)));
                    }

                    return $tTicket;
                }
            }

            return null;
        }

        /**
         * Default Index Action
         */
        public function index() {
            $request = $this->getRequest();
            $engine = $this->getView();

            $user_id = $this->user_id;
            $group_id = $this->group_id;

            $engine = $this->getView();
            $currentTicket = $this->currentTicket;

            // Umleiten zur Historie
            if ($request->post('historie') != '') {
                Dispatcher::route("/historie/");
            }

            // finde das aktuelle Ticket, welches per GET live angegeben wurde.
            if ($request->get('tid') != '') {
                $currentTicket = $this->processTickedId($user_id);
                $this->handleRequest($currentTicket, $user_id, $group_id);

                return;
            }

            // lade das ausgewählte Ticket, Supporter nehmen es automatisch an.
            if ($request->post('laden') != '') {
                $currentTicket = $this->processLaden($user_id);
                $this->handleRequest($currentTicket, $user_id, $group_id);

                return;
            }

            // erstelle oder aktualisiere das aktuelle Ticket.
            if ($request->post('eintragen') != '') {
                $currentTicket = $this->processEintragen($user_id, $group_id);
                $this->handleRequest($currentTicket, $user_id, $group_id);

                return;
            }

            // lösche das aktuelle Ticket.
            if ($request->post('loeschen') != '') {
                $this->processLoeschen($user_id);
                $this->handleRequest($currentTicket, $user_id, $group_id);

                return;
            }

            $this->handleRequest($this->currentTicket, $this->user_id, $this->group_id);
        }

    }
}