<?php

/**
 * Copyright(C) 2015 Markus Bröker<broeker.markus@googlemail.com>
 *
 */

namespace bfw;

use Logger;

/**
 * Class Request
 */
class Request {

    protected $logger = null;

    public function __construct() {
        $this->logger = Logger::getLogger('__CLASS__');

        $this->initSession();
    }

    /**
     * <b>Initialisiert die Session für das jeweilige Projekt</b>
     *
     *
     */
    public function initSession() {
        if (!isset($_SESSION['prefix'])) {
            $_SESSION['prefix'] = array(
                'get' => array(),
                'session' => array(),
                'isLoggedIn' => false,
                'user_id' => 1,
                'group_id' => 1,
                'error' => '',
                'msg' => '',
                'lastpage' => 'home',
            );
        }
    }

    /**
     * Liefert die komplette URL des jeweiligen Requests
     *
     * @return string
     */
    public static function getUrl() {
        return sprintf("%s%s", self::getBaseUrl(), $_SERVER['REQUEST_URI']);
    }

    /**
     * Liefert die Basis-URL des jeweiligen Projekts
     *
     * @return string
     */
    public static function getBaseUrl() {
        return sprintf("%s://%s", isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] != 'off' ? 'https' : 'http', $_SERVER['SERVER_NAME']);
    }

    /**
     * <b>Liefert einen gefilterten 'session' Wert
     *
     * Hierbei handelt es sich um einen zuvor gespeicherten POST-Wert,
     * der in ['prefix']['session'] gelandet ist...
     *
     * @param $param
     * @param string $default
     * @return string
     */
    public static function session($param, $default = '') {
        if (isset($_SESSION['prefix']['session'][$param])) {
            return $_SESSION['prefix']['session'][$param];
        }

        return $default;
    }

    /**
     * <b>Liefert einen gefilterten 'get' Wert
     *
     * Hierbei handelt es sich um einen zuvor gespeicherten POST-Wert,
     * der in ['prefix']['session'] geladet ist...
     *
     * @param $param
     * @param string $default
     * @return string
     */
    public static function getParameter($param, $default = '') {
        if (isset($_SESSION['prefix']['get'][$param])) {
            return $_SESSION['prefix']['get'][$param];
        }

        return $default;
    }

    /**
     * <b>Holt einen Session Wert aus dem jeweiligen Prefix</b>
     *
     * @param $key
     * @param string $default
     * @param null $storage
     * @return string
     */
    public static function getKey($key, $default = '', $storage = null) {
        if ($storage != null) {
            return $_SESSION['prefix'][$storage][$key];
        }

        if (isset($_SESSION['prefix'][$key])) {
            return $_SESSION['prefix'][$key];
        }

        return $default;
    }

    /**
     * <b>Löscht einen Get-Wert aus dem jeweiligen Prefix</b>
     *
     * @param $key
     */
    public static function deleteParameterKey($key) {
        unset($_SESSION['prefix']['get'][$key]);
    }

    /**
     * <b>Löscht einen Session Wert aus dem jeweiligen Prefix</b>
     *
     * @param $key
     */
    public static function deleteKey($key) {
        unset($_SESSION['prefix'][$key]);
    }

    /**
     * <b>Prüft, ob der aktuelle Request einen POST enthält</b>
     *
     * @return bool
     */
    public static function hasPost() {
        return (count($_POST) != 0);
    }

    /**
     * Keine Warnung, keine Gnade
     */
    public function destroySession() {
        unset($_SESSION['prefix']);
        @session_destroy();

        $this->initSession();
    }

    /**
     * <b>Bei jedem Seitenaufruf werden die GET/Post Anfragen in der Session separiert gespeichert</b>
     *
     * Struktur:
     *    GET Sachen ins 'get' array
     *   POST Sachen ins 'session' array
     */
    public function keepRequestData() {
        foreach ($_GET as $key => $value) {
            $filteredKey = $this->filter($key);
            $filteredValue = $this->filter($value);

            $_SESSION['prefix']['get'][$filteredKey] = $filteredValue;
        }

        foreach ($_POST as $key => $value) {
            $filteredKey = $this->filter($key);
            $filteredValue = $this->filter($value);

            $_SESSION['prefix']['session'][$filteredKey] = $filteredValue;
        }
    }

    /**
     * <b>Statische Filtermethode für beliebige Strings/Array of Strings</b>
     *
     * filter_input kann nur mit GET/POST umgehen, diese Methode mit Allem.
     *
     * @param string $param
     * @return string|array
     */
    public static function filter($param) {
        if (is_array($param)) {
            $filtered = array();
            foreach ($param as $key => $value) {
                $filtered[self::filter($key)] = self::filter($value);
            }

            return $filtered;
        }

        return htmlspecialchars($param, ENT_QUOTES, 'utf-8');
    }

    /**
     * <b>Setzt einen neuen Session Wert ins jeweilige Prefix</b>
     *
     * @param $key
     * @param $value
     */
    public function setKey($key, $value, $storage = null) {
        if ($storage != null) {
            $_SESSION['prefix'][$storage][self::filter($key)] = self::filter($value);
        } else {
            $_SESSION['prefix'][self::filter($key)] = self::filter($value);
        }
    }

    /**
     * <b>Interne Debug-Funktion zum Anzeigen der Session Struktur</b>
     *
     * Beliebige URL mit ?debug=1 aufrufen und der Debug Mechanismus ist aktiviert.
     *
     * @return string
     */
    public function printSession() {
        $this->logger->info(sprintf('SESSION: %s', print_r($_SESSION['prefix'], true)));

        if ($this->get('debug') == '') {
            return '';
        }

        return "<br /><pre>" . print_r($_SESSION['prefix'], true) . "</pre>";
    }

    /**
     * <b>Liefert einen gefilterten GET-Wert</b>
     *
     * @param $param
     * @return string
     */
    public function get($param, $default = '') {
        $value = filter_input(INPUT_GET, $param);

        if ($value == '') {
            return $default;
        }

        return $value;
    }

    /**
     * <b>Liefert einen gefilterten POST-Wert</b>
     *
     * @param $param
     * @return string
     */
    public function post($param, $default = '') {
        $value = filter_input(INPUT_POST, $param);

        if ($value == '') {
            return $default;
        }

        return $value;
    }

}
