/**
 *   $Id: ChessBoard.java 152 2008-04-27 02:04:31Z mbroeker $
 *  $URL: http://localhost/svn/eclipse/Schachspiel/trunk/org/homelinux/largo/games/board/chessboard/ChessBoard.java $
 */

package org.homelinux.largo.games.board.chessboard;

import java.awt.Image;

import org.homelinux.largo.games.board.Board;
import org.homelinux.largo.games.board.Piece;
import org.homelinux.largo.utils.ImgComponent;

public class ChessBoard extends Board {
    static final long serialVersionUID = 1L;

    static final int ROOK = 1;
    static final int KNIGHT = 2;
    static final int BISHOP = 3;
    static final int QUEEN = 4;
    static final int KING = 5;
    static final int PAWN = 6;

    final Piece white_queen = new Piece(new ImgComponent("images/white_queen.png").getImage(), "white", QUEEN, 90);
    final Piece black_queen = new Piece(new ImgComponent("images/black_queen.png").getImage(), "black", QUEEN, 90);

    boolean white_rochades_small;
    boolean white_rochades_big;
    boolean black_rochades_small;
    boolean black_rochades_big;

    void initBoard() {
        add(new ImgComponent("images/black_rook.png").getImage(), 0, "black", ROOK);
        add(new ImgComponent("images/black_knight.png").getImage(), 1, "black", KNIGHT);
        add(new ImgComponent("images/black_bishop.png").getImage(), 2, "black", BISHOP);
        add(new ImgComponent("images/black_queen.png").getImage(), 3, "black", QUEEN);
        add(new ImgComponent("images/black_king.png").getImage(), 4, "black", KING);
        add(new ImgComponent("images/black_bishop.png").getImage(), 5, "black", BISHOP);
        add(new ImgComponent("images/black_knight.png").getImage(), 6, "black", KNIGHT);
        add(new ImgComponent("images/black_rook.png").getImage(), 7, "black", ROOK);

        for (int i = 0; i < 8; i++)
            add(new ImgComponent("images/black_pawn.png").getImage(), 8 + i, "black", PAWN);

        add(new ImgComponent("images/white_rook.png").getImage(), 56, "white", ROOK);
        add(new ImgComponent("images/white_knight.png").getImage(), 57, "white", KNIGHT);
        add(new ImgComponent("images/white_bishop.png").getImage(), 58, "white", BISHOP);
        add(new ImgComponent("images/white_queen.png").getImage(), 59, "white", QUEEN);
        add(new ImgComponent("images/white_king.png").getImage(), 60, "white", KING);
        add(new ImgComponent("images/white_bishop.png").getImage(), 61, "white", BISHOP);
        add(new ImgComponent("images/white_knight.png").getImage(), 62, "white", KNIGHT);
        add(new ImgComponent("images/white_rook.png").getImage(), 63, "white", ROOK);

        for (int i = 0; i < 8; i++)
            add(new ImgComponent("images/white_pawn.png").getImage(), 55 - i, "white", PAWN);
    }

    /**
     * The init method initializes a complete chess board with figures and a
     * history.
     */
    public void init() {
        super.init();
        initBoard();
        board_pieces = getPieces();
        repaint();
    }

    public ChessBoard(int w, int h) {
        super(w, h);
    }

    void add(Image img, int i, String c, int t) {
        int v = 0;

        switch (t) {
        case ROOK:
            v = 80;
            break;
        case KNIGHT:
            v = 30;
            break;
        case BISHOP:
            v = 30;
            break;
        case QUEEN:
            v = 90;
            break;
        case KING:
            v = 120;
            break;
        case PAWN:
            v = 10;
            break;
        }

        setPiece(i, new Piece(img, c, t, v));
    }

    /**
     * returns true, when the white king is at pieces[i]
     */
    public boolean isWhiteKing(int i) {
        if (isWhite(i) && getType(i) == KING)
            return true;
        return false;
    }

    /**
     * returns true, when the black king is at pieces[i]
     */
    public boolean isBlackKing(int i) {
        if (isBlack(i) && getType(i) == KING)
            return true;
        return false;
    }

    /**
     * Returns the score of the current position Every field controlled by
     * [color] gives one point A King, controlled by [color], gives 10 extra
     * points
     */
    public int controls(String color) {
        int t, o;
        int value = 0;

        for (o = 0; o < 64; o++) {
            if (color.equals(getColor(o))) {
                for (t = 0; t < 64; t++) {
                    if (validMove(t, o)) {
                        if (getType(t) == KING)
                            value += 10;
                        value++;
                    }
                }
            }
        }
        return value;
    }

    /**
     * Checks, wether this move is possible or not.
     */
    public boolean validMove(int t, int o) {
        int steps;
        int rows;
        int dx;

        /* Must be here */
        white_rochades_small = false;
        white_rochades_big = false;
        black_rochades_small = false;
        black_rochades_big = false;

        if (t == o)
            return false;

        if (isSamePiece(t, o))
            return false;

        if (getType(o) == EMPTY)
            return false;

        /*
         * 00 01 02 03 04 05 06 07
         * 08 09 10 11 12 13 14 15
         * 16 17 18 19 20 21 22 23
         * 24 25 26 27 28 29 30 31
         * 32 33 34 35 36 37 38 39
         * 40 41 42 43 44 45 46 47
         * 48 49 50 51 52 53 54 55
         * 56 57 58 59 60 61 62 63
         */

        steps = Math.abs(t - o);
        rows = Math.abs(t / 8 - o / 8);

        switch (getType(o)) {
        case ROOK:
            if (steps < 8 && rows == 0) {
                if (t > o)
                    for (dx = o + 1; dx < t; dx++) {
                        if (getColor(dx) != null) {
                            return false;
                        }
                    }
                else
                    for (dx = o - 1; dx > t; dx--) {
                        if (getColor(dx) != null) {
                            return false;
                        }
                    }
                return true;
            }
            if (steps % 8 == 0) {
                if (t > o) {
                    for (dx = o + 8; dx < t; dx += 8) {
                        if (getColor(dx) != null)
                            return false;
                    }
                } else {
                    for (dx = o - 8; dx > t; dx -= 8) {
                        if (getColor(dx) != null)
                            return false;
                    }
                }
                if (steps == 8 && rows == 1)
                    return true;
                if (steps == 16 && rows == 2)
                    return true;
                if (steps == 24 && rows == 3)
                    return true;
                if (steps == 32 && rows == 4)
                    return true;
                if (steps == 40 && rows == 5)
                    return true;
                if (steps == 48 && rows == 6)
                    return true;
                if (steps == 56 && rows == 7)
                    return true;
            }
            break;
        case KNIGHT:
            /* works fine */
            if (steps == 6 && rows == 1)
                return true;
            if (steps == 10 && rows == 1)
                return true;
            if (steps == 15 && rows == 2)
                return true;
            if (steps == 17 && rows == 2)
                return true;
            break;
        case BISHOP:
            if (steps % 7 == 0) {
                if (t > o) {
                    for (dx = o + 7; dx < t; dx += 7) {
                        if (getColor(dx) != null)
                            return false;
                    }
                } else {
                    for (dx = o - 7; dx > t; dx -= 7) {
                        if (getColor(dx) != null)
                            return false;
                    }
                }
                if (steps == 7 && rows == 1)
                    return true;
                if (steps == 14 && rows == 2)
                    return true;
                if (steps == 21 && rows == 3)
                    return true;
                if (steps == 28 && rows == 4)
                    return true;
                if (steps == 35 && rows == 5)
                    return true;
                if (steps == 42 && rows == 6)
                    return true;
                if (steps == 49 && rows == 7)
                    return true;
            }
            if (steps % 9 == 0) {
                if (t > o) {
                    for (dx = o + 9; dx < t; dx += 9) {
                        if (getColor(dx) != null)
                            return false;
                    }
                } else {
                    for (dx = o - 9; dx > t; dx -= 9) {
                        if (getColor(dx) != null)
                            return false;
                    }
                }
                if (steps == 9 && rows == 1)
                    return true;
                if (steps == 18 && rows == 2)
                    return true;
                if (steps == 27 && rows == 3)
                    return true;
                if (steps == 36 && rows == 4)
                    return true;
                if (steps == 45 && rows == 5)
                    return true;
                if (steps == 54 && rows == 6)
                    return true;
                if (steps == 63 && rows == 7)
                    return true;
            }
            break;

        case QUEEN:
            if (steps < 8 && rows == 0) {
                if (t > o)
                    for (dx = o + 1; dx < t; dx++) {
                        if (getColor(dx) != null)
                            return false;
                    }
                else
                    for (dx = o - 1; dx > t; dx--) {
                        if (getColor(dx) != null)
                            return false;
                    }
                return true;
            }

            if (steps % 8 == 0) {
                if (t > o) {
                    for (dx = o + 8; dx < t; dx += 8) {
                        if (getColor(dx) != null)
                            return false;
                    }
                } else {
                    for (dx = o - 8; dx > t; dx -= 8) {
                        if (getColor(dx) != null)
                            return false;
                    }
                }

                if (steps == 8 && rows == 1)
                    return true;
                if (steps == 16 && rows == 2)
                    return true;
                if (steps == 24 && rows == 3)
                    return true;
                if (steps == 32 && rows == 4)
                    return true;
                if (steps == 40 && rows == 5)
                    return true;
                if (steps == 48 && rows == 6)
                    return true;
                if (steps == 56 && rows == 7)
                    return true;
            }
            if (steps % 7 == 0) {
                if (t > o) {
                    for (dx = o + 7; dx < t; dx += 7) {
                        if (getColor(dx) != null)
                            return false;
                    }
                } else {
                    for (dx = o - 7; dx > t; dx -= 7) {
                        if (getColor(dx) != null)
                            return false;
                    }
                }
                if (steps == 7 && rows == 1)
                    return true;
                if (steps == 14 && rows == 2)
                    return true;
                if (steps == 21 && rows == 3)
                    return true;
                if (steps == 28 && rows == 4)
                    return true;
                if (steps == 35 && rows == 5)
                    return true;
                if (steps == 42 && rows == 6)
                    return true;
                if (steps == 49 && rows == 7)
                    return true;
            }

            if (steps % 9 == 0) {
                if (t > o) {
                    for (dx = o + 9; dx < t; dx += 9) {
                        if (getColor(dx) != null)
                            return false;
                    }
                } else {
                    for (dx = o - 9; dx > t; dx -= 9) {
                        if (getColor(dx) != null)
                            return false;
                    }
                }
                if (steps == 9 && rows == 1)
                    return true;
                if (steps == 18 && rows == 2)
                    return true;
                if (steps == 27 && rows == 3)
                    return true;
                if (steps == 36 && rows == 4)
                    return true;
                if (steps == 45 && rows == 5)
                    return true;
                if (steps == 54 && rows == 6)
                    return true;
                if (steps == 63 && rows == 7)
                    return true;
            }
            break;

        case KING:
            if ((steps == 1) && (rows == 0))
                return true;

            if ((steps == 8) && (rows == 1))
                return true;

            if ((steps == 7) && (rows == 1))
                return true;

            if ((steps == 9) && (rows == 1))
                return true;

            if ((steps == 2) && (rows == 0)) {
                if (isWhiteKing(o)) { /* White: Rochade */
                    if (isWhiteKing(60) && isEmpty(61) && isEmpty(62) && (t == 62)) {
                        if ((getType(63) == ROOK) && isWhite(63)) {
                            white_rochades_small = true;
                            return true;
                        }
                    }

                    if (isWhiteKing(60) && isEmpty(59) && isEmpty(58) && isEmpty(57) && (t == 58)) {
                        if ((getType(56) == ROOK) && isWhite(56)) {
                            white_rochades_big = true;
                            return true;
                        }
                    }
                }

                if (isBlackKing(o)) { /* Black: Rochade */
                    if (isBlackKing(4) && isEmpty(5) && isEmpty(6) && (t == 6)) {
                        if ((getType(7) == ROOK) && isBlack(7)) {
                            black_rochades_small = true;
                            return true;
                        }
                    }

                    if (isBlackKing(4) && isEmpty(3) && isEmpty(2) && isEmpty(1) && (t == 2)) {
                        if ((getType(0) == ROOK) && isBlack(0)) {
                            black_rochades_big = true;
                            return true;
                        }
                    }
                }
            }
            break;

        case PAWN:
            if (getColor(o).equals("white")) {
                if (steps == 7 && rows == 1 && isEnemy(t, o) && t < o)
                    return true;

                if (steps == 9 && rows == 1 && isEnemy(t, o) && t < o)
                    return true;
            } else {
                if (steps == 7 && rows == 1 && isEnemy(t, o) && t > o)
                    return true;

                if (steps == 9 && rows == 1 && isEnemy(t, o) && t > o)
                    return true;
            }

            if (steps % 8 == 0) {
                if (t > o) {
                    for (dx = o + 8; dx <= t; dx += 8) {
                        if (getColor(dx) != null)
                            return false;
                    }
                } else {
                    for (dx = o - 8; dx >= t; dx -= 8) {
                        if (getColor(dx) != null)
                            return false;
                    }
                }

                if (steps == 8 && rows == 1) {
                    if (getColor(o).equals("white") && t < o)
                        return true;
                    if (getColor(o).equals("black") && t > o)
                        return true;
                }

                if (steps == 16 && rows == 2 && (o >= 8 && o <= 15)) {
                    if (getColor(o).equals("white") && t < o)
                        return true;
                    if (getColor(o).equals("black") && t > o)
                        return true;
                }

                if (steps == 16 && rows == 2 && (o >= 48 && o <= 55)) {
                    if (getColor(o).equals("white") && t < o)
                        return true;
                    if (getColor(o).equals("black") && t > o)
                        return true;
                }
            }
            break;
        }
        return false;
    }

    /**
     * This method must be called immediately after isValidMove(t, o)
     */
    boolean rochade() {
        if (white_rochades_small) {
            setPiece(61, getPiece(63));
            setPiece(63, new Piece(null, null, EMPTY, 0));
            return true;
        }

        if (white_rochades_big) {
            setPiece(59, getPiece(56));
            setPiece(56, new Piece(null, null, EMPTY, 0));
            return true;
        }

        if (black_rochades_small) {
            setPiece(5, getPiece(7));
            setPiece(7, new Piece(null, null, EMPTY, 0));
            return true;
        }

        if (black_rochades_big) {
            setPiece(3, getPiece(0));
            setPiece(0, new Piece(null, null, EMPTY, 0));
            return true;
        }

        return false;
    }

    /**
     * checks, whether color is in check or not.
     */
    public boolean isCheck(String color) {
        int i;
        int wking;
        int bking;

        wking = bking = -1;

        for (i = 0; i < 64; i++) {
            if (isWhiteKing(i)) {
                wking = i;
            }
            if (isBlackKing(i)) {
                bking = i;
            }
        }

        // returns 64 false positives
        if (wking == -1 || bking == -1)
            return true;

        if (color.equals("white")) {
            for (i = 0; i < 64; i++)
                if (validMove(wking, i)) {
                    return true;
                }

            return false;
        }

        if (color.equals("black")) {
            for (i = 0; i < 64; i++)
                if (validMove(bking, i)) {
                    return true;
                }

            return false;
        }
        return false;
    }

    /**
     * simulates a "valid" move or returns false. all changes are relative to
     * pieces and will not be painted. This method sets internal variables for
     * the rochade() method, which will be overwritten.
     */
    public boolean simulateMove(int t, int o) {
        if (!validMove(t, o)) {
            return false;
        }

        push(getPiece(t), t, getPiece(o), o);

        if (getType(o) == PAWN) {
            if (isBlack(o) && t >= 56)
                setPiece(o, new Piece(black_queen));

            if (isWhite(o) && t <= 7)
                setPiece(o, new Piece(white_queen));
        }

        setPiece(t, new Piece(getPiece(o)));
        setPiece(o, new Piece(null, null, EMPTY, 0));

        if (isCheck(getColor(t))) {
            /* restore current position */
            if (UNDO_DEBUG)
                print("UNDO", t, o, 0);
            undo();
            return false;
        }

        return true;
    }

    /**
     * performs a "valid" move or returns false. all changes are relative to
     * board_pieces and will be painted.
     */
    public boolean doMove(int t, int o) {
        if (!validMove(t, o)) {
            return false;
        }

        push(getPiece(t), t, getPiece(o), o);

        if (getType(o) == PAWN) {
            if (isBlack(o) && t >= 56)
                setPiece(o, new Piece(black_queen));

            if (isWhite(o) && t <= 7)
                setPiece(o, new Piece(white_queen));
        }

        setPiece(t, new Piece(getPiece(o)));
        setPiece(o, new Piece(null, null, EMPTY, 0));

        if (rochade())
            System.out.println("Rochade");

        if (isCheck(getColor(t))) {
            // restore current positions
            if (UNDO_DEBUG)
                print("UNDO", t, o, 0);
            undo();
            return false;
        }

        board_pieces = getPieces();
        repaint();

        moveNr = stack.size();

        return true;
    }

    /**
     * converts the internal representation into normal chess notation and
     * writes it to stdout
     */
    public void print(String s, int t, int o, int value) {
        int row1, col1, row2, col2;

        row1 = o / 8;
        col1 = o - (8 * row1);

        row2 = t / 8;
        col2 = t - (8 * row2);

        System.out.printf("%s: %C%d-%c%d = %3d%n", s, ('A') + col1, 8 - row1, ('A') + col2, 8 - row2, value);
    }
}
